import { Link } from 'react-router-dom';
import { ArrowRight, Play } from 'lucide-react';
import ScrollReveal from '@/components/ScrollReveal';
import ProjectCard from '@/components/ProjectCard';
import Layout from '@/components/Layout';

const projects = [
  {
    title: 'The Art of Motion',
    category: 'Brand Film',
    image: 'https://images.unsplash.com/photo-1536440136628-849c177e76a1?w=800&q=80',
    slug: 'art-of-motion',
  },
  {
    title: 'Urban Horizons',
    category: 'Documentary',
    image: 'https://images.unsplash.com/photo-1485846234645-a62644f84728?w=800&q=80',
    slug: 'urban-horizons',
  },
  {
    title: 'Essence of Light',
    category: 'Commercial',
    image: 'https://images.unsplash.com/photo-1518676590629-3dcbd9c5a5c9?w=800&q=80',
    slug: 'essence-of-light',
  },
  {
    title: 'Beyond Boundaries',
    category: 'Music Video',
    image: 'https://images.unsplash.com/photo-1574717024653-61fd2cf4d44d?w=800&q=80',
    slug: 'beyond-boundaries',
  },
  {
    title: 'Silent Stories',
    category: 'Short Film',
    image: 'https://images.unsplash.com/photo-1492691527719-9d1e07e534b4?w=800&q=80',
    slug: 'silent-stories',
  },
  {
    title: 'Neon Dreams',
    category: 'Fashion Film',
    image: 'https://images.unsplash.com/photo-1516035069371-29a1b244cc32?w=800&q=80',
    slug: 'neon-dreams',
  },
];

const Index = () => {
  return (
    <Layout>
      {/* Hero Section */}
      <section className="relative min-h-screen flex items-center justify-center overflow-hidden">
        {/* Background Video/Image */}
        <div className="absolute inset-0">
          <img
            src="https://images.unsplash.com/photo-1478720568477-152d9b164e26?w=1920&q=80"
            alt="Hero"
            className="w-full h-full object-cover scale-in"
          />
          <div className="absolute inset-0 bg-foreground/40" />
        </div>

        {/* Hero Content */}
        <div className="relative z-10 text-center text-background px-6">
          <div className="overflow-hidden mb-6">
            <h1 className="text-5xl md:text-7xl lg:text-8xl font-light tracking-tight fade-up">
              We Create
            </h1>
          </div>
          <div className="overflow-hidden mb-8">
            <h1 className="text-5xl md:text-7xl lg:text-8xl font-light tracking-tight fade-up-delay-1">
              <span className="italic">Visual</span> Stories
            </h1>
          </div>
          <p className="text-lg md:text-xl text-background/80 max-w-2xl mx-auto mb-12 fade-up-delay-2">
            Premium film & video production company crafting compelling narratives
            for brands that dare to stand out.
          </p>
          <div className="flex flex-col sm:flex-row items-center justify-center gap-4 fade-up-delay-3">
            <Link
              to="/work"
              className="group inline-flex items-center gap-2 bg-background text-foreground px-8 py-4 text-sm uppercase tracking-wider hover:bg-background/90 transition-all duration-300"
            >
              View Our Work
              <ArrowRight className="w-4 h-4 group-hover:translate-x-1 transition-transform" />
            </Link>
            <button className="group inline-flex items-center gap-2 border border-background text-background px-8 py-4 text-sm uppercase tracking-wider hover:bg-background hover:text-foreground transition-all duration-300">
              <Play className="w-4 h-4" />
              Play Showreel
            </button>
          </div>
        </div>

        {/* Scroll Indicator */}
        <div className="absolute bottom-12 left-1/2 -translate-x-1/2 fade-up-delay-4">
          <div className="flex flex-col items-center gap-2 text-background/60">
            <span className="text-xs uppercase tracking-widest">Scroll</span>
            <div className="w-px h-12 bg-background/30 relative overflow-hidden">
              <div className="absolute top-0 left-0 w-full h-1/2 bg-background animate-bounce" />
            </div>
          </div>
        </div>
      </section>

      {/* Introduction Section */}
      <section className="py-24 md:py-32 bg-background">
        <div className="container mx-auto px-6">
          <div className="max-w-4xl mx-auto text-center">
            <ScrollReveal>
              <p className="text-xs uppercase tracking-[0.3em] text-muted-foreground mb-6">
                About Go Productions
              </p>
            </ScrollReveal>
            <ScrollReveal delay={100}>
              <h2 className="text-3xl md:text-4xl lg:text-5xl font-light leading-relaxed mb-8">
                We believe in the power of <span className="italic">visual storytelling</span> to transform brands and captivate audiences worldwide.
              </h2>
            </ScrollReveal>
            <ScrollReveal delay={200}>
              <Link
                to="/about"
                className="inline-flex items-center gap-2 text-sm uppercase tracking-wider hover:gap-4 transition-all duration-300"
              >
                Learn More About Us
                <ArrowRight className="w-4 h-4" />
              </Link>
            </ScrollReveal>
          </div>
        </div>
      </section>

      {/* Featured Projects */}
      <section className="py-24 md:py-32 bg-secondary">
        <div className="container mx-auto px-6">
          <ScrollReveal>
            <div className="flex flex-col md:flex-row md:items-end md:justify-between gap-6 mb-16">
              <div>
                <p className="text-xs uppercase tracking-[0.3em] text-muted-foreground mb-4">
                  Selected Works
                </p>
                <h2 className="text-4xl md:text-5xl font-light">
                  Featured Projects
                </h2>
              </div>
              <Link
                to="/work"
                className="inline-flex items-center gap-2 text-sm uppercase tracking-wider hover:gap-4 transition-all duration-300"
              >
                View All Projects
                <ArrowRight className="w-4 h-4" />
              </Link>
            </div>
          </ScrollReveal>

          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-8 md:gap-12">
            {projects.map((project, index) => (
              <ScrollReveal key={project.slug} delay={index * 100}>
                <ProjectCard {...project} index={index} />
              </ScrollReveal>
            ))}
          </div>
        </div>
      </section>

      {/* Services Section */}
      <section className="py-24 md:py-32 bg-background">
        <div className="container mx-auto px-6">
          <ScrollReveal>
            <div className="text-center mb-16">
              <p className="text-xs uppercase tracking-[0.3em] text-muted-foreground mb-4">
                Our Expertise
              </p>
              <h2 className="text-4xl md:text-5xl font-light">
                What We Do
              </h2>
            </div>
          </ScrollReveal>

          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-8">
            {[
              { title: 'Film Production', desc: 'End-to-end production services for commercials, documentaries, and brand films.' },
              { title: 'Cinematography', desc: 'Stunning visuals captured by our expert cinematographers using cutting-edge equipment.' },
              { title: 'Post Production', desc: 'Color grading, editing, and visual effects to bring your vision to life.' },
              { title: 'Creative Direction', desc: 'Strategic creative guidance from concept development to final delivery.' },
            ].map((service, index) => (
              <ScrollReveal key={service.title} delay={index * 100}>
                <div className="group p-8 border border-border hover:border-foreground hover:bg-foreground hover:text-background transition-all duration-500">
                  <span className="text-xs text-muted-foreground group-hover:text-background/60 mb-4 block">
                    0{index + 1}
                  </span>
                  <h3 className="text-xl font-light mb-4">{service.title}</h3>
                  <p className="text-sm text-muted-foreground group-hover:text-background/70 leading-relaxed">
                    {service.desc}
                  </p>
                </div>
              </ScrollReveal>
            ))}
          </div>
        </div>
      </section>

      {/* CTA Section */}
      <section className="py-24 md:py-32 bg-foreground text-background">
        <div className="container mx-auto px-6">
          <ScrollReveal>
            <div className="max-w-4xl mx-auto text-center">
              <h2 className="text-4xl md:text-5xl lg:text-6xl font-light mb-8">
                Have a project in mind?
              </h2>
              <p className="text-lg text-background/70 mb-12 max-w-2xl mx-auto">
                Let's collaborate to create something extraordinary. We're always excited to work with brands that share our passion for visual excellence.
              </p>
              <Link
                to="/contact"
                className="inline-flex items-center gap-2 bg-background text-foreground px-8 py-4 text-sm uppercase tracking-wider hover:bg-background/90 transition-all duration-300"
              >
                Get In Touch
                <ArrowRight className="w-4 h-4" />
              </Link>
            </div>
          </ScrollReveal>
        </div>
      </section>
    </Layout>
  );
};

export default Index;
