import { useState } from 'react';
import { Send, MapPin, Phone, Mail, Clock } from 'lucide-react';
import Layout from '@/components/Layout';
import ScrollReveal from '@/components/ScrollReveal';
import { Input } from '@/components/ui/input';
import { Textarea } from '@/components/ui/textarea';
import { Button } from '@/components/ui/button';

const Contact = () => {
  const [formData, setFormData] = useState({
    name: '',
    email: '',
    phone: '',
    company: '',
    message: '',
  });

  const handleSubmit = (e: React.FormEvent) => {
    e.preventDefault();
    // Form submission logic would go here
    console.log('Form submitted:', formData);
  };

  const handleChange = (e: React.ChangeEvent<HTMLInputElement | HTMLTextAreaElement>) => {
    setFormData({
      ...formData,
      [e.target.name]: e.target.value,
    });
  };

  const contactInfo = [
    {
      icon: MapPin,
      title: 'Visit Us',
      lines: ['Silver Oak Farms', 'Chhatarpur, New Delhi', 'India - 110074'],
    },
    {
      icon: Phone,
      title: 'Call Us',
      lines: ['+91 98765 43210', '+91 11 4567 8900'],
    },
    {
      icon: Mail,
      title: 'Email Us',
      lines: ['info@goproductions.in', 'projects@goproductions.in'],
    },
    {
      icon: Clock,
      title: 'Working Hours',
      lines: ['Monday - Friday', '10:00 AM - 7:00 PM IST'],
    },
  ];

  return (
    <Layout>
      {/* Hero Section */}
      <section className="pt-32 pb-16 md:pt-40 md:pb-24 bg-background">
        <div className="container mx-auto px-6">
          <div className="max-w-4xl">
            <ScrollReveal>
              <p className="text-xs uppercase tracking-[0.3em] text-muted-foreground mb-4">
                Get In Touch
              </p>
            </ScrollReveal>
            <ScrollReveal delay={100}>
              <h1 className="text-5xl md:text-6xl lg:text-7xl font-light mb-6">
                Let's Create Together
              </h1>
            </ScrollReveal>
            <ScrollReveal delay={200}>
              <p className="text-lg text-muted-foreground max-w-2xl">
                Have a project in mind? We'd love to hear from you. Send us a message 
                and we'll respond as soon as possible.
              </p>
            </ScrollReveal>
          </div>
        </div>
      </section>

      {/* Contact Form & Image Section */}
      <section className="py-16 md:py-24 bg-background">
        <div className="container mx-auto px-6">
          <div className="grid grid-cols-1 lg:grid-cols-2 gap-12 lg:gap-24">
            {/* Form */}
            <ScrollReveal>
              <form onSubmit={handleSubmit} className="space-y-6">
                <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                  <div className="space-y-2">
                    <label htmlFor="name" className="text-sm uppercase tracking-wider">
                      Name *
                    </label>
                    <Input
                      id="name"
                      name="name"
                      value={formData.name}
                      onChange={handleChange}
                      placeholder="Your name"
                      required
                      className="border-0 border-b border-border rounded-none bg-transparent px-0 py-3 focus-visible:ring-0 focus-visible:border-foreground transition-colors"
                    />
                  </div>
                  <div className="space-y-2">
                    <label htmlFor="email" className="text-sm uppercase tracking-wider">
                      Email *
                    </label>
                    <Input
                      id="email"
                      name="email"
                      type="email"
                      value={formData.email}
                      onChange={handleChange}
                      placeholder="your@email.com"
                      required
                      className="border-0 border-b border-border rounded-none bg-transparent px-0 py-3 focus-visible:ring-0 focus-visible:border-foreground transition-colors"
                    />
                  </div>
                </div>

                <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                  <div className="space-y-2">
                    <label htmlFor="phone" className="text-sm uppercase tracking-wider">
                      Phone
                    </label>
                    <Input
                      id="phone"
                      name="phone"
                      value={formData.phone}
                      onChange={handleChange}
                      placeholder="+91 XXXXX XXXXX"
                      className="border-0 border-b border-border rounded-none bg-transparent px-0 py-3 focus-visible:ring-0 focus-visible:border-foreground transition-colors"
                    />
                  </div>
                  <div className="space-y-2">
                    <label htmlFor="company" className="text-sm uppercase tracking-wider">
                      Company
                    </label>
                    <Input
                      id="company"
                      name="company"
                      value={formData.company}
                      onChange={handleChange}
                      placeholder="Your company"
                      className="border-0 border-b border-border rounded-none bg-transparent px-0 py-3 focus-visible:ring-0 focus-visible:border-foreground transition-colors"
                    />
                  </div>
                </div>

                <div className="space-y-2">
                  <label htmlFor="message" className="text-sm uppercase tracking-wider">
                    Message *
                  </label>
                  <Textarea
                    id="message"
                    name="message"
                    value={formData.message}
                    onChange={handleChange}
                    placeholder="Tell us about your project..."
                    required
                    rows={6}
                    className="border-0 border-b border-border rounded-none bg-transparent px-0 py-3 focus-visible:ring-0 focus-visible:border-foreground transition-colors resize-none"
                  />
                </div>

                <Button
                  type="submit"
                  className="w-full md:w-auto bg-foreground text-background hover:bg-foreground/90 rounded-none px-12 py-6 text-sm uppercase tracking-wider"
                >
                  Send Message
                  <Send className="w-4 h-4 ml-2" />
                </Button>
              </form>
            </ScrollReveal>

            {/* Aesthetic Image */}
            <ScrollReveal delay={200}>
              <div className="relative h-full min-h-[400px] lg:min-h-0">
                <img
                  src="https://images.unsplash.com/photo-1485846234645-a62644f84728?w=800&q=80"
                  alt="Production studio"
                  className="w-full h-full object-cover"
                />
                <div className="absolute inset-0 bg-foreground/20" />
                
                {/* Floating text */}
                <div className="absolute bottom-8 left-8 right-8 text-background">
                  <p className="text-3xl md:text-4xl font-light">
                    Every great project starts with a conversation.
                  </p>
                </div>
              </div>
            </ScrollReveal>
          </div>
        </div>
      </section>

      {/* Contact Info Cards */}
      <section className="py-16 md:py-24 bg-secondary">
        <div className="container mx-auto px-6">
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-8">
            {contactInfo.map((info, index) => (
              <ScrollReveal key={info.title} delay={index * 100}>
                <div className="bg-background p-8 h-full">
                  <info.icon className="w-6 h-6 mb-4" />
                  <h3 className="text-sm uppercase tracking-wider font-medium mb-4">
                    {info.title}
                  </h3>
                  <div className="space-y-1">
                    {info.lines.map((line, i) => (
                      <p key={i} className="text-muted-foreground">
                        {line}
                      </p>
                    ))}
                  </div>
                </div>
              </ScrollReveal>
            ))}
          </div>
        </div>
      </section>

      {/* Map Section */}
      <section className="h-[500px] md:h-[600px] bg-muted relative">
        <iframe
          src="https://www.google.com/maps/embed?pb=!1m18!1m12!1m3!1d3505.5651508626943!2d77.17494611508214!3d28.510889982464997!2m3!1f0!2f0!3f0!3m2!1i1024!2i768!4f13.1!3m3!1m2!1s0x390d1e0f6b4a8c57%3A0x5c6f97d5b3ef9f9a!2sChhatarpur%2C%20New%20Delhi%2C%20Delhi!5e0!3m2!1sen!2sin!4v1699900000000!5m2!1sen!2sin"
          width="100%"
          height="100%"
          style={{ border: 0, filter: 'grayscale(100%) contrast(1.1)' }}
          allowFullScreen
          loading="lazy"
          referrerPolicy="no-referrer-when-downgrade"
          title="Go Productions Location"
        />
        
        {/* Map Overlay Card */}
        <div className="absolute bottom-8 left-8 bg-background p-6 md:p-8 max-w-sm">
          <h3 className="text-lg font-light mb-2">Go Productions</h3>
          <p className="text-sm text-muted-foreground">
            Silver Oak Farms, Chhatarpur<br />
            New Delhi, India - 110074
          </p>
          <a
            href="https://maps.google.com/?q=Chhatarpur,New+Delhi"
            target="_blank"
            rel="noopener noreferrer"
            className="inline-block mt-4 text-sm uppercase tracking-wider hover:underline"
          >
            Get Directions →
          </a>
        </div>
      </section>
    </Layout>
  );
};

export default Contact;
