// Go Productions - Main JavaScript

// ===== CUSTOM CURSOR =====
class CustomCursor {
  constructor() {
    // Check if device is mobile/touch
    this.isTouch = 'ontouchstart' in window || navigator.maxTouchPoints > 0;

    if (this.isTouch) return;

    this.cursor = document.createElement('div');
    this.cursor.className = 'custom-cursor';
    this.follower = document.createElement('div');
    this.follower.className = 'custom-cursor-follower';
    document.body.appendChild(this.cursor);
    document.body.appendChild(this.follower);

    this.cursorPos = { x: 0, y: 0 };
    this.followerPos = { x: 0, y: 0 };

    this.init();
  }

  init() {
    if (this.isTouch) return;
    // Track mouse position
    document.addEventListener('mousemove', (e) => {
      this.cursorPos.x = e.clientX;
      this.cursorPos.y = e.clientY;
    });

    // Animate cursor and follower
    const animate = () => {
      // Main cursor follows directly
      this.cursor.style.left = this.cursorPos.x + 'px';
      this.cursor.style.top = this.cursorPos.y + 'px';

      // Follower has slight delay for smooth effect
      this.followerPos.x += (this.cursorPos.x - this.followerPos.x) * 0.15;
      this.followerPos.y += (this.cursorPos.y - this.followerPos.y) * 0.15;

      this.follower.style.left = this.followerPos.x + 'px';
      this.follower.style.top = this.followerPos.y + 'px';

      requestAnimationFrame(animate);
    };
    animate();

    // Add active class on clickable elements
    const clickables = document.querySelectorAll('a, button, input, textarea, [role="button"], .project-card, .client-card');
    clickables.forEach(el => {
      el.addEventListener('mouseenter', () => {
        this.cursor.classList.add('active');
        this.follower.style.opacity = '0';
      });
      el.addEventListener('mouseleave', () => {
        this.cursor.classList.remove('active');
        this.follower.style.opacity = '1';
      });
    });
  }
}

// ===== HEADER SCROLL =====
class HeaderScroll {
  constructor() {
    this.header = document.querySelector('header');
    this.init();
  }

  init() {
    window.addEventListener('scroll', () => {
      if (window.scrollY > 50) {
        this.header.classList.add('scrolled');
      } else {
        this.header.classList.remove('scrolled');
      }
    });
  }
}

// ===== MOBILE MENU =====
class MobileMenu {
  constructor() {
    this.menuBtn = document.querySelector('.mobile-menu-btn');
    this.mobileMenu = document.querySelector('.mobile-menu');
    this.init();
  }

  init() {
    if (!this.menuBtn || !this.mobileMenu) return;

    // Create close button if it doesn't exist
    if (!this.mobileMenu.querySelector('.close-btn')) {
      const closeBtn = document.createElement('button');
      closeBtn.className = 'close-btn';
      closeBtn.innerHTML = '&times;';
      this.mobileMenu.appendChild(closeBtn);
    }

    this.closeBtn = this.mobileMenu.querySelector('.close-btn');

    // Toggle Menu
    this.menuBtn.addEventListener('click', () => {
      this.mobileMenu.classList.add('active');
      document.body.style.overflow = 'hidden';
    });

    // Close Menu
    this.closeBtn.addEventListener('click', () => {
      this.mobileMenu.classList.remove('active');
      document.body.style.overflow = '';
    });

    // Close menu when clicking on a link
    const links = this.mobileMenu.querySelectorAll('a');
    links.forEach(link => {
      link.addEventListener('click', () => {
        this.mobileMenu.classList.remove('active');
        document.body.style.overflow = '';
      });
    });
  }
}

// ===== SCROLL REVEAL =====
class ScrollReveal {
  constructor() {
    this.elements = document.querySelectorAll('[data-reveal]');
    this.init();
  }

  init() {
    if ('IntersectionObserver' in window) {
      const observer = new IntersectionObserver((entries) => {
        entries.forEach(entry => {
          if (entry.isIntersecting) {
            entry.target.classList.add('fade-up');
            observer.unobserve(entry.target);
          }
        });
      }, {
        threshold: 0.1,
        rootMargin: '0px 0px -50px 0px'
      });

      this.elements.forEach(el => observer.observe(el));
    } else {
      // Fallback for browsers without IntersectionObserver
      this.elements.forEach(el => el.classList.add('fade-up'));
    }
  }
}

// ===== ACTIVE NAV LINK =====
class ActiveNavLink {
  constructor() {
    this.init();
  }

  init() {
    const currentPage = window.location.pathname.split('/').pop() || 'index.html';
    const navLinks = document.querySelectorAll('nav a');

    navLinks.forEach(link => {
      const href = link.getAttribute('href');
      if (href === currentPage || (currentPage === '' && href === 'index.html')) {
        link.classList.add('active');
      }
    });
  }
}

// ===== WORK FILTER =====
class WorkFilter {
  constructor() {
    this.filterBtns = document.querySelectorAll('.filter-btn');
    this.projectCards = document.querySelectorAll('.project-card');
    this.init();
  }

  init() {
    if (this.filterBtns.length === 0) return;

    this.filterBtns.forEach(btn => {
      btn.addEventListener('click', () => {
        const category = btn.dataset.category;

        // Update active button
        this.filterBtns.forEach(b => b.classList.remove('active'));
        btn.classList.add('active');

        // Filter projects
        this.projectCards.forEach(card => {
          if (category === 'all' || card.dataset.category === category) {
            card.style.display = 'block';
            setTimeout(() => card.style.opacity = '1', 10);
          } else {
            card.style.opacity = '0';
            setTimeout(() => card.style.display = 'none', 300);
          }
        });
      });
    });
  }
}

// ===== CONTACT FORM =====
class ContactForm {
  constructor() {
    this.form = document.querySelector('.contact-form');
    this.init();
  }

  init() {
    if (!this.form) return;

    this.form.addEventListener('submit', (e) => {
      e.preventDefault();

      // Get form data
      const formData = new FormData(this.form);
      const data = Object.fromEntries(formData);

      // You would typically send this to a server
      console.log('Form submitted:', data);

      // Show success message
      alert('Thank you for your message! We will get back to you soon.');
      this.form.reset();
    });
  }
}

// ===== SMOOTH SCROLL =====
class SmoothScroll {
  constructor() {
    this.init();
  }

  init() {
    document.querySelectorAll('a[href^="#"]').forEach(anchor => {
      anchor.addEventListener('click', function (e) {
        e.preventDefault();
        const target = document.querySelector(this.getAttribute('href'));
        if (target) {
          target.scrollIntoView({
            behavior: 'smooth',
            block: 'start'
          });
        }
      });
    });
  }
}

// ===== IMAGE LAZY LOADING =====
class LazyLoad {
  constructor() {
    this.images = document.querySelectorAll('img[data-src]');
    this.init();
  }

  init() {
    if ('IntersectionObserver' in window) {
      const imageObserver = new IntersectionObserver((entries) => {
        entries.forEach(entry => {
          if (entry.isIntersecting) {
            const img = entry.target;
            img.src = img.dataset.src;
            img.removeAttribute('data-src');
            imageObserver.unobserve(img);
          }
        });
      });

      this.images.forEach(img => imageObserver.observe(img));
    } else {
      // Fallback
      this.images.forEach(img => {
        img.src = img.dataset.src;
        img.removeAttribute('data-src');
      });
    }
  }
}

// ===== INITIALIZE ALL =====
document.addEventListener('DOMContentLoaded', () => {
  // Initialize all components
  new CustomCursor();
  new HeaderScroll();
  new MobileMenu();
  new ScrollReveal();
  new ActiveNavLink();
  new WorkFilter();
  new ContactForm();
  new SmoothScroll();
  new LazyLoad();

  // Add page enter animation
  document.body.style.opacity = '0';
  setTimeout(() => {
    document.body.style.transition = 'opacity 0.6s ease';
    document.body.style.opacity = '1';
  }, 100);
});

// ===== EXPORT FOR MODULES (if needed) =====
if (typeof module !== 'undefined' && module.exports) {
  module.exports = {
    CustomCursor,
    HeaderScroll,
    MobileMenu,
    ScrollReveal,
    ActiveNavLink,
    WorkFilter,
    ContactForm,
    SmoothScroll,
    LazyLoad
  };
}
